/*
 * Decompiled with CFR 0.152.
 */
package com.zaxxer.hikari.pool;

import com.codahale.metrics.MetricRegistry;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.IConnectionCustomizer;
import com.zaxxer.hikari.metrics.CodaHaleMetricsTracker;
import com.zaxxer.hikari.metrics.MetricsTracker;
import com.zaxxer.hikari.pool.HikariMBeanElf;
import com.zaxxer.hikari.pool.HikariPoolMBean;
import com.zaxxer.hikari.pool.PoolBagEntry;
import com.zaxxer.hikari.proxy.IHikariConnectionProxy;
import com.zaxxer.hikari.proxy.ProxyFactory;
import com.zaxxer.hikari.util.ConcurrentBag;
import com.zaxxer.hikari.util.DefaultThreadFactory;
import com.zaxxer.hikari.util.LeakTask;
import com.zaxxer.hikari.util.PoolUtilities;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HikariPool
implements HikariPoolMBean,
ConcurrentBag.IBagStateListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(HikariPool.class);
    private static final long ALIVE_BYPASS_WINDOW = Long.getLong("com.zaxxer.hikari.aliveBypassWindow", 1000L);
    public final String catalog;
    public final boolean isReadOnly;
    public final boolean isAutoCommit;
    public int transactionIsolation;
    private final DataSource dataSource;
    private final HikariConfig configuration;
    private final MetricsTracker metricsTracker;
    private final ThreadPoolExecutor addConnectionExecutor;
    private final ConcurrentBag<PoolBagEntry> connectionBag;
    private final ThreadPoolExecutor closeConnectionExecutor;
    private final IConnectionCustomizer connectionCustomizer;
    private final AtomicInteger totalConnections;
    private final AtomicReference<Throwable> lastConnectionFailure;
    private final ScheduledThreadPoolExecutor houseKeepingExecutorService;
    private final String username;
    private final String password;
    private final boolean isRecordMetrics;
    private final boolean isIsolateInternalQueries;
    private final long leakDetectionThreshold;
    private volatile boolean isShutdown;
    private volatile long connectionTimeout;
    private volatile boolean isUseJdbc4Validation;

    public HikariPool(HikariConfig configuration) {
        this(configuration, configuration.getUsername(), configuration.getPassword());
    }

    public HikariPool(HikariConfig configuration, String username, String password) {
        this.username = username;
        this.password = password;
        this.configuration = configuration;
        this.connectionBag = new ConcurrentBag(this);
        this.totalConnections = new AtomicInteger();
        this.connectionTimeout = configuration.getConnectionTimeout();
        this.lastConnectionFailure = new AtomicReference();
        this.isReadOnly = configuration.isReadOnly();
        this.isAutoCommit = configuration.isAutoCommit();
        this.catalog = configuration.getCatalog();
        this.connectionCustomizer = this.initializeCustomizer();
        this.transactionIsolation = PoolUtilities.getTransactionIsolation(configuration.getTransactionIsolation());
        this.leakDetectionThreshold = configuration.getLeakDetectionThreshold();
        this.isIsolateInternalQueries = configuration.isIsolateInternalQueries();
        this.isRecordMetrics = configuration.getMetricRegistry() != null;
        this.metricsTracker = this.isRecordMetrics ? new CodaHaleMetricsTracker(this, (MetricRegistry)configuration.getMetricRegistry()) : new MetricsTracker(this);
        this.dataSource = PoolUtilities.initializeDataSource(configuration.getDataSourceClassName(), configuration.getDataSource(), configuration.getDataSourceProperties(), configuration.getJdbcUrl(), username, password);
        PoolUtilities.setLoginTimeout(this.dataSource, this.connectionTimeout, LOGGER);
        HikariMBeanElf.registerMBeans(configuration, this);
        this.addConnectionExecutor = PoolUtilities.createThreadPoolExecutor(configuration.getMaximumPoolSize(), "HikariCP connection filler (pool " + configuration.getPoolName() + ")", configuration.getThreadFactory(), new ThreadPoolExecutor.DiscardPolicy());
        this.closeConnectionExecutor = PoolUtilities.createThreadPoolExecutor(4, "HikariCP connection closer (pool " + configuration.getPoolName() + ")", configuration.getThreadFactory(), new ThreadPoolExecutor.CallerRunsPolicy());
        long delayPeriod = Long.getLong("com.zaxxer.hikari.housekeeping.periodMs", TimeUnit.SECONDS.toMillis(30L));
        this.houseKeepingExecutorService = new ScheduledThreadPoolExecutor(1, configuration.getThreadFactory() != null ? configuration.getThreadFactory() : new DefaultThreadFactory("Hikari Housekeeping Timer (pool " + configuration.getPoolName() + ")", true));
        this.houseKeepingExecutorService.setRemoveOnCancelPolicy(true);
        this.houseKeepingExecutorService.scheduleAtFixedRate(new HouseKeeper(), delayPeriod, delayPeriod, TimeUnit.MILLISECONDS);
        this.fillPool();
    }

    public Connection getConnection() throws SQLException {
        long timeout = this.connectionTimeout;
        long start = System.currentTimeMillis();
        MetricsTracker.MetricsContext metricsContext = this.isRecordMetrics ? this.metricsTracker.recordConnectionRequest(start) : MetricsTracker.NO_CONTEXT;
        try {
            do {
                PoolBagEntry bagEntry;
                if ((bagEntry = this.connectionBag.borrow(timeout, TimeUnit.MILLISECONDS)) == null) {
                    break;
                }
                long now = System.currentTimeMillis();
                if (now <= bagEntry.expirationTime && (now - bagEntry.lastAccess <= ALIVE_BYPASS_WINDOW || this.isConnectionAlive(bagEntry.connection, timeout))) {
                    LeakTask leakTask = this.leakDetectionThreshold == 0L ? LeakTask.NO_LEAK : new LeakTask(this.leakDetectionThreshold, this.houseKeepingExecutorService);
                    IHikariConnectionProxy proxyConnection = ProxyFactory.getProxyConnection(this, bagEntry, leakTask);
                    metricsContext.setConnectionLastOpen(bagEntry, now);
                    IHikariConnectionProxy iHikariConnectionProxy = proxyConnection;
                    return iHikariConnectionProxy;
                }
                this.closeConnection(bagEntry);
                timeout = this.connectionTimeout - PoolUtilities.elapsedTimeMs(start);
            } while (timeout > 0L);
        }
        catch (InterruptedException e) {
            throw new SQLException("Interrupted during connection acquisition", e);
        }
        finally {
            metricsContext.stop();
        }
        this.logPoolState("Timeout failure ");
        throw new SQLException(String.format("Timeout after %dms of waiting for a connection.", PoolUtilities.elapsedTimeMs(start)), this.lastConnectionFailure.getAndSet(null));
    }

    public void releaseConnection(PoolBagEntry bagEntry, boolean isBroken) {
        this.metricsTracker.recordConnectionUsage(bagEntry);
        if (isBroken || bagEntry.evicted) {
            LOGGER.debug("Connection returned to pool {} is broken or evicted.  Closing connection.", (Object)this.configuration.getPoolName());
            this.closeConnection(bagEntry);
            return;
        }
        bagEntry.lastAccess = System.currentTimeMillis();
        this.connectionBag.requite(bagEntry);
    }

    public void shutdown() throws InterruptedException {
        if (!this.isShutdown) {
            this.isShutdown = true;
            LOGGER.info("HikariCP pool {} is shutting down.", (Object)this.configuration.getPoolName());
            this.connectionBag.close();
            this.houseKeepingExecutorService.shutdownNow();
            this.addConnectionExecutor.shutdownNow();
            this.logPoolState("Before shutdown ");
            long start = System.currentTimeMillis();
            do {
                this.softEvictConnections();
                this.abortActiveConnections();
            } while ((this.getIdleConnections() > 0 || this.getActiveConnections() > 0) && PoolUtilities.elapsedTimeMs(start) < TimeUnit.SECONDS.toMillis(5L));
            this.closeConnectionExecutor.shutdown();
            this.closeConnectionExecutor.awaitTermination(5L, TimeUnit.SECONDS);
            this.logPoolState("After shutdown ");
            HikariMBeanElf.unregisterMBeans(this.configuration, this);
        }
    }

    public void evictConnection(IHikariConnectionProxy proxyConnection) {
        this.closeConnection(proxyConnection.getPoolBagEntry());
    }

    public DataSource getDataSource() {
        return this.dataSource;
    }

    public HikariConfig getConfiguration() {
        return this.configuration;
    }

    public String toString() {
        return this.configuration.getPoolName();
    }

    @Override
    public void addBagItem() {
        this.addConnectionExecutor.submit(() -> {
            long sleepBackoff = 200L;
            int maxPoolSize = this.configuration.getMaximumPoolSize();
            int minIdle = this.configuration.getMinimumIdle();
            while (!(this.isShutdown || this.totalConnections.get() >= maxPoolSize || minIdle != 0 && this.getIdleConnections() >= minIdle)) {
                if (this.addConnection()) {
                    if (minIdle != 0) continue;
                    break;
                }
                if (minIdle == 0 && this.getThreadsAwaitingConnection() == 0) break;
                PoolUtilities.quietlySleep(sleepBackoff);
                sleepBackoff = Math.min(this.connectionTimeout / 2L, (long)((double)sleepBackoff * 1.5));
            }
        });
    }

    @Override
    public int getActiveConnections() {
        return this.connectionBag.getCount(1);
    }

    @Override
    public int getIdleConnections() {
        return this.connectionBag.getCount(0);
    }

    @Override
    public int getTotalConnections() {
        return this.connectionBag.size() - this.connectionBag.getCount(-1);
    }

    @Override
    public int getThreadsAwaitingConnection() {
        return this.connectionBag.getPendingQueue();
    }

    @Override
    public void softEvictConnections() {
        this.connectionBag.values(1).forEach(bagEntry -> {
            bagEntry.evicted = true;
        });
        this.connectionBag.values(0).stream().filter(p -> this.connectionBag.reserve((PoolBagEntry)p)).forEach(bagEntry -> this.closeConnection((PoolBagEntry)bagEntry));
    }

    private void closeConnection(PoolBagEntry bagEntry) {
        int tc = this.totalConnections.decrementAndGet();
        this.connectionBag.remove(bagEntry);
        if (tc < 0) {
            LOGGER.warn("Internal accounting inconsistency, totalConnections={}", (Object)tc, (Object)new Exception());
        }
        this.closeConnectionExecutor.submit(() -> PoolUtilities.quietlyCloseConnection(poolBagEntry.connection));
    }

    private boolean addConnection() {
        if (this.totalConnections.incrementAndGet() > this.configuration.getMaximumPoolSize() || this.isShutdown) {
            this.totalConnections.decrementAndGet();
            return true;
        }
        Connection connection = null;
        try {
            connection = this.username == null && this.password == null ? this.dataSource.getConnection() : this.dataSource.getConnection(this.username, this.password);
            boolean bl = this.isUseJdbc4Validation = PoolUtilities.isJdbc40Compliant(connection) && this.configuration.getConnectionTestQuery() == null;
            if (!this.isUseJdbc4Validation && this.configuration.getConnectionTestQuery() == null) {
                LOGGER.error("JDBC4 Connection.isValid() method not supported, connection test query must be configured");
            }
            boolean timeoutEnabled = this.connectionTimeout != Integer.MAX_VALUE;
            long timeoutMs = timeoutEnabled ? Math.max(250L, this.connectionTimeout) : 0L;
            int originalTimeout = PoolUtilities.setNetworkTimeout(this.houseKeepingExecutorService, connection, timeoutMs, timeoutEnabled);
            this.transactionIsolation = this.transactionIsolation < 0 ? connection.getTransactionIsolation() : this.transactionIsolation;
            PoolUtilities.setupConnection(connection, this.isAutoCommit, this.isReadOnly, this.transactionIsolation, this.catalog);
            this.connectionCustomizer.customize(connection);
            PoolUtilities.executeSql(connection, this.configuration.getConnectionInitSql(), this.isAutoCommit);
            PoolUtilities.setNetworkTimeout(this.houseKeepingExecutorService, connection, originalTimeout, timeoutEnabled);
            this.connectionBag.add(new PoolBagEntry(connection, this.configuration.getMaxLifetime()));
            this.lastConnectionFailure.set(null);
            return true;
        }
        catch (Exception e) {
            this.totalConnections.decrementAndGet();
            this.lastConnectionFailure.set(e);
            PoolUtilities.quietlyCloseConnection(connection);
            LOGGER.debug("Connection attempt to database {} failed: {}", this.configuration.getPoolName(), e.getMessage(), e);
            return false;
        }
    }

    private boolean isConnectionAlive(Connection connection, long timeoutMs) {
        try {
            int timeoutSec;
            boolean timeoutEnabled = this.connectionTimeout != Integer.MAX_VALUE;
            int n = timeoutSec = timeoutEnabled ? (int)Math.max(1L, TimeUnit.MILLISECONDS.toSeconds(timeoutMs)) : 0;
            if (this.isUseJdbc4Validation) {
                return connection.isValid(timeoutSec);
            }
            int networkTimeout = PoolUtilities.setNetworkTimeout(this.houseKeepingExecutorService, connection, Math.max(250, (int)timeoutMs), timeoutEnabled);
            try (Statement statement = connection.createStatement();){
                PoolUtilities.setQueryTimeout(statement, timeoutSec);
                statement.executeQuery(this.configuration.getConnectionTestQuery());
            }
            if (this.isIsolateInternalQueries && !this.isAutoCommit) {
                connection.rollback();
            }
            PoolUtilities.setNetworkTimeout(this.houseKeepingExecutorService, connection, networkTimeout, timeoutEnabled);
            return true;
        }
        catch (SQLException e) {
            LOGGER.warn("Exception during keep alive check, that means the connection ({}) must be dead.", (Object)connection, (Object)e);
            return false;
        }
    }

    private void fillPool() {
        if (this.configuration.getMinimumIdle() > 0) {
            if (this.configuration.isInitializationFailFast() && !this.addConnection()) {
                throw new RuntimeException("Fail-fast during pool initialization", this.lastConnectionFailure.getAndSet(null));
            }
            this.addBagItem();
        }
    }

    private void abortActiveConnections() throws InterruptedException {
        ThreadPoolExecutor assassinExecutor = PoolUtilities.createThreadPoolExecutor(this.configuration.getMaximumPoolSize(), "HikariCP connection assassin", this.configuration.getThreadFactory(), new ThreadPoolExecutor.CallerRunsPolicy());
        this.connectionBag.values(1).stream().forEach(bagEntry -> {
            try {
                bagEntry.connection.abort(assassinExecutor);
            }
            catch (AbstractMethodError | SQLException e) {
                PoolUtilities.quietlyCloseConnection(bagEntry.connection);
            }
            finally {
                try {
                    this.connectionBag.remove((PoolBagEntry)bagEntry);
                    this.totalConnections.decrementAndGet();
                }
                catch (IllegalStateException ise) {
                    return;
                }
            }
        });
        assassinExecutor.shutdown();
        assassinExecutor.awaitTermination(5L, TimeUnit.SECONDS);
    }

    private IConnectionCustomizer initializeCustomizer() {
        if (this.configuration.getConnectionCustomizerClassName() != null) {
            return PoolUtilities.createInstance(this.configuration.getConnectionCustomizerClassName(), IConnectionCustomizer.class, new Object[0]);
        }
        return this.configuration.getConnectionCustomizer();
    }

    public void logPoolState(String ... prefix) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("{}pool stats {} (total={}, inUse={}, avail={}, waiting={})", prefix.length > 0 ? prefix[0] : "", this.configuration.getPoolName(), this.getTotalConnections(), this.getActiveConnections(), this.getIdleConnections(), this.getThreadsAwaitingConnection());
        }
    }

    private class HouseKeeper
    implements Runnable {
        private HouseKeeper() {
        }

        @Override
        public void run() {
            HikariPool.this.logPoolState("Before cleanup ");
            HikariPool.this.connectionTimeout = HikariPool.this.configuration.getConnectionTimeout();
            long now = System.currentTimeMillis();
            long idleTimeout = HikariPool.this.configuration.getIdleTimeout();
            HikariPool.this.connectionBag.values(0).stream().filter(p -> HikariPool.this.connectionBag.reserve(p)).forEach(bagEntry -> {
                if (idleTimeout > 0L && now > bagEntry.lastAccess + idleTimeout || now > bagEntry.expirationTime) {
                    HikariPool.this.closeConnection(bagEntry);
                } else {
                    HikariPool.this.connectionBag.unreserve(bagEntry);
                }
            });
            HikariPool.this.logPoolState("After cleanup ");
            if (HikariPool.this.configuration.getMinimumIdle() > 0) {
                HikariPool.this.addBagItem();
            }
        }
    }
}

